# 导入xml.etree.ElementTree库，用于处理XML文件
import xml.etree.ElementTree as ET
# 导入os库，用于处理文件和目录
import os
# 导入PIL库的Image模块，用于处理图像
from PIL import Image

# 定义一个函数parse_xml，参数为xml文件的路径
def parse_xml(xml_file) -> dict:
    # 解析xml文件，获取ElementTree对象
    tree = ET.parse(xml_file)
    # 获取xml文件的根元素
    root = tree.getroot()

    # 初始化一个空字典来存储数据
    data = {}
    # 从xml文件中获取各个元素的值，并存储到字典中
    data['folder'] = root.find('folder').text
    data['filename'] = root.find('filename').text
    data['width'] = root.find('size/width').text
    data['height'] = root.find('size/height').text
    data['depth'] = root.find('size/depth').text
    data['segmented'] = root.find('segmented').text

    # 初始化一个空列表来存储所有的对象数据
    objects = []
    # 遍历xml文件中的所有'object'元素
    for obj in root.findall('object'):
        # 初始化一个空字典来存储当前对象的数据
        obj_data = {}
        # 从'object'元素中获取各个子元素的值，并存储到字典中
        obj_data['name'] = obj.find('name').text
        obj_data['pose'] = obj.find('pose').text
        obj_data['truncated'] = obj.find('truncated').text
        obj_data['occluded'] = obj.find('occluded').text
        obj_data['difficult'] = obj.find('difficult').text
        obj_data['xmin'] = obj.find('bndbox/xmin').text
        obj_data['ymin'] = obj.find('bndbox/ymin').text
        obj_data['xmax'] = obj.find('bndbox/xmax').text
        obj_data['ymax'] = obj.find('bndbox/ymax').text
        # 将当前对象的数据添加到对象列表中
        objects.append(obj_data)

    # 将对象列表添加到数据字典中
    data['objects'] = objects

    # 返回数据字典
    return data

# 定义一个函数get_coordinates，参数为从parse_xml函数返回的数据字典
def get_coordinates(data:dict) -> tuple:
    # 获取数据字典中的第一个对象
    obj = data['objects'][0]
    # 返回对象的xmin, ymin, xmax, ymax值（转换为整数）
    return int(obj['xmin']), int(obj['ymin']), int(obj['xmax']), int(obj['ymax'])

# 定义一个函数crop_image，参数为图片路径、坐标和新图片路径
def crop_image(image_path:str, coordinates:tuple,new_image_path:str) -> None:
    # 打开图片
    img = Image.open(image_path)
    # 根据给定的坐标裁剪图片
    cropped_img = img.crop(coordinates)
    # 保存裁剪后的图片到新的路径
    cropped_img.save(new_image_path)

def split_bndbox():
    # 遍历'annotations'目录下的所有文件
    for annotations in os.listdir('annotations'):
        # 构建annotations文件的完整路径
        annotations_path = 'annotations/' + annotations
        '''
        通过观察可以发现，描述文件和图片存在一一对应的关系，仅仅是 文件夹名 与 文件拓展名 不同：
        annotations\Cars0.xml
        images\Cars0.png
        '''
        # 构建对应的图片文件路径
        image_path = annotations_path.replace('xml', 'png').replace('annotations', 'images')
        # 构建裁剪后的新图片路径
        new_image_path = os.path.join('cropped_images',annotations.replace('xml', 'png'))
        # 如果'cropped_images'目录不存在，则创建
        if not os.path.exists('cropped_images'):
            os.makedirs('cropped_images')
        # 解析annotations文件，获取数据
        data = parse_xml('annotations/' + annotations)
        # 从数据中获取坐标
        coordinates = get_coordinates(data)
        # 根据坐标裁剪图片，并保存到新的路径
        crop_image(image_path, coordinates , new_image_path)

if __name__ == '__main__':
    split_bndbox()